/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.resource.loader;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Function;
import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.fabricmc.fabric.api.resource.ResourceManagerHelper;
import net.fabricmc.fabric.api.resource.ResourcePackActivationType;
import net.fabricmc.fabric.api.resource.v1.ResourceLoader;
import net.fabricmc.loader.api.ModContainer;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3262;
import net.minecraft.class_3264;
import net.minecraft.class_3288;
import net.minecraft.class_3302;
import net.minecraft.class_3545;
import net.minecraft.class_7225;
import net.minecraft.class_8614;
import net.minecraft.class_9224;
import net.minecraft.class_9225;

public class ResourceManagerHelperImpl implements ResourceManagerHelper {
	private static final Map<class_3264, ResourceManagerHelperImpl> registryMap = new HashMap<>();
	private static final Set<class_3545<class_2561, ModNioResourcePack>> builtinResourcePacks = new HashSet<>();

	private final ResourceLoader resourceLoader;

	private ResourceManagerHelperImpl(class_3264 type) {
		this.resourceLoader = ResourceLoader.get(type);
	}

	public static ResourceManagerHelperImpl get(class_3264 type) {
		return registryMap.computeIfAbsent(type, ResourceManagerHelperImpl::new);
	}

	/**
	 * Registers a built-in resource pack. Internal implementation.
	 *
	 * @param id             the identifier of the resource pack
	 * @param subPath        the sub path in the mod resources
	 * @param container      the mod container
	 * @param displayName    the display name of the resource pack
	 * @param activationType the activation type of the resource pack
	 * @return {@code true} if successfully registered the resource pack, else {@code false}
	 * @see ResourceManagerHelper#registerBuiltinResourcePack(class_2960, ModContainer, class_2561, ResourcePackActivationType)
	 * @see ResourceManagerHelper#registerBuiltinResourcePack(class_2960, ModContainer, ResourcePackActivationType)
	 */
	public static boolean registerBuiltinResourcePack(class_2960 id, String subPath, ModContainer container, class_2561 displayName, ResourcePackActivationType activationType) {
		// Assuming the mod has multiple paths, we simply "hope" that the  file separator is *not* different across them
		List<Path> paths = container.getRootPaths();
		String separator = paths.getFirst().getFileSystem().getSeparator();
		subPath = subPath.replace("/", separator);
		ModNioResourcePack resourcePack = ModNioResourcePack.create(id.toString(), container, subPath, class_3264.field_14188, activationType, false);
		ModNioResourcePack dataPack = ModNioResourcePack.create(id.toString(), container, subPath, class_3264.field_14190, activationType, false);
		if (resourcePack == null && dataPack == null) return false;

		if (resourcePack != null) {
			builtinResourcePacks.add(new class_3545<>(displayName, resourcePack));
		}

		if (dataPack != null) {
			builtinResourcePacks.add(new class_3545<>(displayName, dataPack));
		}

		return true;
	}

	/**
	 * Registers a built-in resource pack. Internal implementation.
	 *
	 * @param id             the identifier of the resource pack
	 * @param subPath        the sub path in the mod resources
	 * @param container      the mod container
	 * @param activationType the activation type of the resource pack
	 * @return {@code true} if successfully registered the resource pack, else {@code false}
	 * @see ResourceManagerHelper#registerBuiltinResourcePack(class_2960, ModContainer, ResourcePackActivationType)
	 * @see ResourceManagerHelper#registerBuiltinResourcePack(class_2960, ModContainer, class_2561, ResourcePackActivationType)
	 */
	public static boolean registerBuiltinResourcePack(class_2960 id, String subPath, ModContainer container, ResourcePackActivationType activationType) {
		return registerBuiltinResourcePack(id, subPath, container, class_2561.method_43470(id.method_12836() + "/" + id.method_12832()), activationType);
	}

	public static void registerBuiltinResourcePacks(class_3264 resourceType, Consumer<class_3288> consumer) {
		// Loop through each registered built-in resource packs and add them if valid.
		for (class_3545<class_2561, ModNioResourcePack> entry : builtinResourcePacks) {
			ModNioResourcePack pack = entry.method_15441();

			// Add the built-in pack only if namespaces for the specified resource type are present.
			if (!pack.method_14406(resourceType).isEmpty()) {
				// Make the resource pack profile for built-in pack, should never be always enabled.
				class_9224 info = new class_9224(
						entry.method_15441().method_14409(),
						entry.method_15442(),
						new BuiltinModResourcePackSource(pack.getFabricModMetadata().getName()),
						entry.method_15441().method_56929()
				);
				class_9225 info2 = new class_9225(
						pack.getActivationType() == ResourcePackActivationType.ALWAYS_ENABLED,
						class_3288.class_3289.field_14280,
						false
				);

				class_3288 profile = class_3288.method_45275(info, new class_3288.class_7680() {
					@Override
					public class_3262 method_52424(class_9224 var1) {
						return entry.method_15441();
					}

					@Override
					public class_3262 method_52425(class_9224 var1, class_3288.class_7679 metadata) {
						ModNioResourcePack pack = entry.method_15441();

						if (metadata.comp_1584().isEmpty()) {
							return pack;
						}

						List<class_3262> overlays = new ArrayList<>(metadata.comp_1584().size());

						for (String overlay : metadata.comp_1584()) {
							overlays.add(pack.createOverlay(overlay));
						}

						return new class_8614(pack, overlays);
					}
				}, resourceType, info2);
				consumer.accept(profile);
			}
		}
	}

	@Override
	public void registerReloadListener(IdentifiableResourceReloadListener listener) {
		this.resourceLoader.registerReloader(listener.getFabricId(), listener);
		listener.getFabricDependencies().forEach(dependency -> this.resourceLoader.addReloaderOrdering(dependency, listener.getFabricId()));
	}

	@Override
	public void registerReloadListener(class_2960 identifier, Function<class_7225.class_7874, IdentifiableResourceReloadListener> listenerFactory) {
		this.resourceLoader.registerReloader(identifier, new class_3302() {
			@Override
			public CompletableFuture<Void> method_25931(class_11558 store, Executor prepareExecutor, class_4045 reloadSynchronizer, Executor applyExecutor) {
				class_7225.class_7874 registries = store.method_72362(ResourceLoader.RELOADER_REGISTRY_LOOKUP_KEY);
				class_3302 resourceReloader = listenerFactory.apply(registries);

				return resourceReloader.method_25931(store, prepareExecutor, reloadSynchronizer, applyExecutor);
			}
		});
	}
}
