/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.indigo.renderer.render;

import java.util.Arrays;
import java.util.function.Function;
import net.fabricmc.fabric.impl.client.indigo.renderer.aocalc.AoLuminanceFix;
import net.minecraft.class_1087;
import net.minecraft.class_128;
import net.minecraft.class_129;
import net.minecraft.class_148;
import net.minecraft.class_1920;
import net.minecraft.class_1921;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_287;
import net.minecraft.class_4076;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4608;
import net.minecraft.class_5819;
import net.minecraft.class_761;

/**
 * Used during section block buffering to invoke {@link class_1087#emitQuads}.
 */
public class TerrainRenderContext extends AbstractTerrainRenderContext {
	public static final ThreadLocal<TerrainRenderContext> POOL = ThreadLocal.withInitial(TerrainRenderContext::new);

	private class_4587 matrixStack;
	private class_5819 random;
	private Function<class_1921, class_287> bufferFunc;

	public TerrainRenderContext() {
		overlay = class_4608.field_21444;
	}

	@Override
	protected LightDataProvider createLightDataProvider(BlockRenderInfo blockInfo) {
		return new LightDataCache(blockInfo);
	}

	@Override
	protected class_4588 getVertexConsumer(class_1921 layer) {
		return bufferFunc.apply(layer);
	}

	public void prepare(class_1920 blockView, class_2338 sectionOrigin, class_4587 matrixStack, class_5819 random, Function<class_1921, class_287> bufferFunc) {
		blockInfo.prepareForWorld(blockView, true);
		((LightDataCache) lightDataProvider).prepare(sectionOrigin);

		this.matrixStack = matrixStack;
		this.random = random;
		this.bufferFunc = bufferFunc;
	}

	public void release() {
		matrices = null;
		matrixStack = null;
		random = null;
		bufferFunc = null;

		blockInfo.release();
	}

	/** Called from section builder hook. */
	public void bufferModel(class_1087 model, class_2680 blockState, class_2338 blockPos) {
		matrixStack.method_22903();

		try {
			matrixStack.method_46416(class_4076.method_18684(blockPos.method_10263()), class_4076.method_18684(blockPos.method_10264()), class_4076.method_18684(blockPos.method_10260()));
			class_243 offset = blockState.method_26226(blockPos);
			matrixStack.method_22904(offset.field_1352, offset.field_1351, offset.field_1350);
			matrices = matrixStack.method_23760();

			random.method_43052(blockState.method_26190(blockPos));

			prepare(blockPos, blockState);
			model.emitQuads(getEmitter(), blockInfo.blockView, blockPos, blockState, random, blockInfo::shouldCullSide);
		} catch (Throwable throwable) {
			class_128 crashReport = class_128.method_560(throwable, "Tessellating block in world - Indigo Renderer");
			class_129 crashReportSection = crashReport.method_562("Block being tessellated");
			class_129.method_586(crashReportSection, blockInfo.blockView, blockPos, blockState);
			throw new class_148(crashReport);
		} finally {
			matrixStack.method_22909();
		}
	}

	private static class LightDataCache implements LightDataProvider {
		// Since this context is only used during section building, we know ahead of time all positions for which data
		// may be requested by flat or smooth lighting, so we use an array instead of a map to cache that data, unlike
		// vanilla. Even though cache indices are positions and therefore 3D, the cache is 1D to maximize memory
		// locality.
		private final int[] lightCache = new int[18 * 18 * 18];
		private final float[] aoCache = new float[18 * 18 * 18];

		private final BlockRenderInfo blockInfo;
		private class_2338 sectionOrigin;

		LightDataCache(BlockRenderInfo blockInfo) {
			this.blockInfo = blockInfo;
		}

		private final class_761.class_10948 lightGetter = (world, pos) -> {
			int cacheIndex = cacheIndex(pos);

			if (cacheIndex == -1) {
				return class_761.class_10948.field_58200.packedBrightness(world, pos);
			}

			int result = lightCache[cacheIndex];

			if (result == Integer.MAX_VALUE) {
				result = class_761.class_10948.field_58200.packedBrightness(world, pos);
				lightCache[cacheIndex] = result;
			}

			return result;
		};

		public void prepare(class_2338 sectionOrigin) {
			this.sectionOrigin = sectionOrigin;

			Arrays.fill(lightCache, Integer.MAX_VALUE);
			Arrays.fill(aoCache, Float.NaN);
		}

		@Override
		public int light(class_2338 pos, class_2680 state) {
			return class_761.method_23793(lightGetter, blockInfo.blockView, state, pos);
		}

		@Override
		public float ao(class_2338 pos, class_2680 state) {
			int cacheIndex = cacheIndex(pos);

			if (cacheIndex == -1) {
				return AoLuminanceFix.INSTANCE.apply(blockInfo.blockView, pos, state);
			}

			float result = aoCache[cacheIndex];

			if (Float.isNaN(result)) {
				result = AoLuminanceFix.INSTANCE.apply(blockInfo.blockView, pos, state);
				aoCache[cacheIndex] = result;
			}

			return result;
		}

		private int cacheIndex(class_2338 pos) {
			int localX = pos.method_10263() - (sectionOrigin.method_10263() - 1);

			if (localX < 0 || localX >= 18) {
				return -1;
			}

			int localY = pos.method_10264() - (sectionOrigin.method_10264() - 1);

			if (localY < 0 || localY >= 18) {
				return -1;
			}

			int localZ = pos.method_10260() - (sectionOrigin.method_10260() - 1);

			if (localZ < 0 || localZ >= 18) {
				return -1;
			}

			return localZ * 18 * 18 + localY * 18 + localX;
		}
	}
}
