/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.client.model.loading.v1;

import net.minecraft.class_1087;
import net.minecraft.class_2248;
import net.minecraft.class_2680;
import org.jetbrains.annotations.ApiStatus;

/**
 * Block state resolvers are responsible for mapping each {@link class_2680} of a block to a
 * {@link class_1087.class_9979}. They replace the {@code blockstates/} JSON files. One block can be mapped to
 * only one block state resolver; multiple resolvers will not receive the same block.
 *
 * <p>Block state resolvers can be used to create custom block state formats or dynamically resolve block state models.
 *
 * <p>Use {@link ModelModifier.OnLoad} instead of this interface if interacting with the block and block states directly
 * is not necessary. Use {@link UnbakedModelDeserializer} for custom model deserializers and loaders.
 *
 * @see ModelModifier.OnLoad
 * @see ModelModifier.OnLoadBlock
 * @see UnbakedModelDeserializer
 */
@FunctionalInterface
public interface BlockStateResolver {
	/**
	 * Resolves the models for all block states of the block.
	 *
	 * <p>For each block state, call {@link Context#setModel} to set its unbaked model.
	 * This method must be called exactly once for each block state.
	 *
	 * <p>Note that if multiple block states share the same unbaked model instance, it will be baked multiple times
	 * (once per block state that has the model set).
	 */
	void resolveBlockStates(Context context);

	/**
	 * The context for block state resolution.
	 */
	@ApiStatus.NonExtendable
	interface Context {
		/**
		 * The block for which block state models are being resolved.
		 */
		class_2248 block();

		/**
		 * Sets the model for a block state.
		 *
		 * @param state the block state for which this model should be used
		 * @param model the unbaked model for this block state
		 */
		void setModel(class_2680 state, class_1087.class_9979 model);
	}
}
