/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.gametest.world;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Properties;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.minecraft.class_1928;
import net.minecraft.class_310;
import net.minecraft.class_3176;
import net.minecraft.class_525;
import net.minecraft.class_5317;
import net.minecraft.class_6880;
import net.minecraft.class_7145;
import net.minecraft.class_7924;
import net.minecraft.class_8100;
import net.minecraft.server.MinecraftServer;
import net.fabricmc.fabric.api.client.gametest.v1.context.ClientGameTestContext;
import net.fabricmc.fabric.api.client.gametest.v1.context.TestDedicatedServerContext;
import net.fabricmc.fabric.api.client.gametest.v1.context.TestSingleplayerContext;
import net.fabricmc.fabric.api.client.gametest.v1.world.TestWorldBuilder;
import net.fabricmc.fabric.impl.client.gametest.context.TestDedicatedServerContextImpl;
import net.fabricmc.fabric.impl.client.gametest.context.TestSingleplayerContextImpl;
import net.fabricmc.fabric.impl.client.gametest.threading.ThreadingImpl;
import net.fabricmc.fabric.impl.client.gametest.util.ClientGameTestImpl;
import net.fabricmc.fabric.impl.client.gametest.util.DedicatedServerImplUtil;

public class TestWorldBuilderImpl implements TestWorldBuilder {
	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-client-gametest-api-v1");
	private final ClientGameTestContext context;
	private boolean useConsistentSettings = true;

	private Consumer<class_8100> settingsAdjustor = creator -> {
	};

	public TestWorldBuilderImpl(ClientGameTestContext context) {
		this.context = context;
	}

	@Override
	public TestWorldBuilder setUseConsistentSettings(boolean useConsistentSettings) {
		this.useConsistentSettings = useConsistentSettings;
		return this;
	}

	@Override
	public TestWorldBuilder adjustSettings(Consumer<class_8100> settingsAdjuster) {
		Preconditions.checkNotNull(settingsAdjuster, "settingsAdjuster");

		this.settingsAdjustor = settingsAdjuster;
		return this;
	}

	@Override
	public TestSingleplayerContext create() {
		ThreadingImpl.checkOnGametestThread("create");
		Preconditions.checkState(!ThreadingImpl.isServerRunning, "Cannot create a world when a server is running");

		Path saveDirectory = navigateCreateWorldScreen();
		ClientGameTestImpl.waitForWorldLoad(context);

		MinecraftServer server = context.computeOnClient(class_310::method_1576);
		return new TestSingleplayerContextImpl(context, new TestWorldSaveImpl(context, saveDirectory), server);
	}

	@Override
	public TestDedicatedServerContext createServer(Properties serverProperties) {
		ThreadingImpl.checkOnGametestThread("createServer");
		Preconditions.checkState(!ThreadingImpl.isServerRunning, "Cannot create a server when a server is running");

		DedicatedServerImplUtil.saveLevelDataTo = Path.of(serverProperties.getProperty("level-name", "world"));

		try {
			FileUtils.deleteDirectory(DedicatedServerImplUtil.saveLevelDataTo.toFile());
		} catch (IOException e) {
			LOGGER.error("Failed to clean up old dedicated server world", e);
		}

		try {
			navigateCreateWorldScreen();
		} finally {
			DedicatedServerImplUtil.saveLevelDataTo = null;
		}

		class_3176 server = DedicatedServerImplUtil.start(context, serverProperties);
		return new TestDedicatedServerContextImpl(context, server);
	}

	private Path navigateCreateWorldScreen() {
		Path saveDirectory = context.computeOnClient(client -> {
			class_525.method_31130(client, client.field_1755);

			if (!(client.field_1755 instanceof class_525 createWorldScreen)) {
				throw new AssertionError("CreateWorldScreen.show did not set the current screen");
			}

			class_8100 creator = createWorldScreen.method_48657();

			if (useConsistentSettings) {
				setConsistentSettings(creator);
			}

			settingsAdjustor.accept(creator);

			return client.method_1586().method_19636().resolve(creator.method_49703());
		});

		context.clickScreenButton("selectWorld.create");

		return saveDirectory;
	}

	private static void setConsistentSettings(class_8100 creator) {
		class_6880<class_7145> flatPreset = creator.method_48728().method_45689().method_30530(class_7924.field_41250).method_46747(class_5317.field_25054);
		creator.method_48705(new class_8100.class_8101(flatPreset));
		creator.method_48716("1");
		creator.method_48717(false);
		creator.method_48734().method_20746(class_1928.field_19396).method_20758(false, null);
		creator.method_48734().method_20746(class_1928.field_19406).method_20758(false, null);
		creator.method_48734().method_20746(class_1928.field_19390).method_20758(false, null);
	}
}
