/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.biome.v1;

import java.util.List;
import java.util.Optional;
import net.minecraft.class_1959;
import net.minecraft.class_2975;
import net.minecraft.class_3195;
import net.minecraft.class_5321;
import net.minecraft.class_5363;
import net.minecraft.class_6796;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_6885;

/**
 * Context given to a biome selector for deciding whether it applies to a biome or not.
 */
public interface BiomeSelectionContext {
	class_5321<class_1959> getBiomeKey();

	/**
	 * Returns the biome with modifications by biome modifiers of higher priority already applied.
	 */
	class_1959 getBiome();

	class_6880<class_1959> getBiomeRegistryEntry();

	/**
	 * Returns true if this biome contains a placed feature referencing a configured feature with the given key.
	 */
	default boolean hasFeature(class_5321<class_2975<?, ?>> key) {
		List<class_6885<class_6796>> featureSteps = getBiome().method_30970().method_30983();

		for (class_6885<class_6796> featureSuppliers : featureSteps) {
			for (class_6880<class_6796> featureSupplier : featureSuppliers) {
				if (featureSupplier.comp_349().method_39643().anyMatch(cf -> getFeatureKey(cf).orElse(null) == key)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Returns true if this biome contains a placed feature with the given key.
	 */
	default boolean hasPlacedFeature(class_5321<class_6796> key) {
		List<class_6885<class_6796>> featureSteps = getBiome().method_30970().method_30983();

		for (class_6885<class_6796> featureSuppliers : featureSteps) {
			for (class_6880<class_6796> featureSupplier : featureSuppliers) {
				if (getPlacedFeatureKey(featureSupplier.comp_349()).orElse(null) == key) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Tries to retrieve the registry key for the given configured feature, which should be from this biomes
	 * current feature list. May be empty if the configured feature is not registered, or does not come
	 * from this biomes feature list.
	 */
	Optional<class_5321<class_2975<?, ?>>> getFeatureKey(class_2975<?, ?> configuredFeature);

	/**
	 * Tries to retrieve the registry key for the given placed feature, which should be from this biomes
	 * current feature list. May be empty if the placed feature is not registered, or does not come
	 * from this biomes feature list.
	 */
	Optional<class_5321<class_6796>> getPlacedFeatureKey(class_6796 placedFeature);

	/**
	 * Returns true if the configured structure with the given key can start in this biome in any chunk generator
	 * used by the current world-save.
	 */
	boolean validForStructure(class_5321<class_3195> key);

	/**
	 * Tries to retrieve the registry key for the given configured feature, which should be from this biomes
	 * current structure list. May be empty if the configured feature is not registered, or does not come
	 * from this biomes feature list.
	 */
	Optional<class_5321<class_3195>> getStructureKey(class_3195 structureFeature);

	/**
	 * Tries to determine whether this biome generates in a specific dimension, based on the {@link net.minecraft.class_5285}
	 * used by the current world-save.
	 *
	 * <p>If no dimension options exist for the given dimension key, <code>false</code> is returned.
	 */
	boolean canGenerateIn(class_5321<class_5363> dimensionKey);

	/**
	 * {@return true if this biome is in the given {@link class_6862}}.
	 */
	boolean hasTag(class_6862<class_1959> tag);
}
